<?php
// app/Http/Controllers/Admin/ContactMessageController.php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ContactMessage;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class ContactMessageController extends Controller
{
    public function index(Request $request)
    {
        $status = $request->query('status');
        $qText  = trim((string) $request->query('q', ''));

        $messages = ContactMessage::query()
            ->when($status !== null && $status !== '', function ($q) use ($status) {
                $q->where('status', $status);
            })
            ->when($qText !== '', function ($q) use ($qText) {
                $q->where(function ($w) use ($qText) {
                    $w->where('name', 'like', "%{$qText}%")
                      ->orWhere('email', 'like', "%{$qText}%")
                      ->orWhere('message', 'like', "%{$qText}%");
                });
            })
            ->latest()
            ->paginate(12)
            ->withQueryString();

        return view('backend.messages.index', compact('messages', 'status'));
    }

    public function show(ContactMessage $cm)
    {
        // Tandai terbaca jika masih "new"
        if ($cm->status === 'new') {
            $cm->update(['status' => 'read']);
        }

        return view('backend.messages.show', compact('cm'));
    }

    public function update(Request $request, ContactMessage $cm)
    {
        // Konsisten dengan tampilan: new | read | resolved
        $data = $request->validate([
            'status' => ['required', Rule::in(['new', 'read', 'resolved'])],
        ]);

        $cm->status = $data['status'];

        // Simpan handler saat dianggap selesai
        if ($cm->status === 'resolved') {
            $cm->handled_by = $request->user()->id ?? null;
            $cm->handled_at = now();
        } else {
            // Jika balik ke new/read, kosongkan handled_* (opsional)
            $cm->handled_by = null;
            $cm->handled_at = null;
        }

        $cm->save();

        return back()->with('success', 'Status pesan diperbarui.');
    }

    public function destroy(ContactMessage $cm)
    {
        $cm->delete();

        return redirect()
            ->route('admin.messages.index')
            ->with('success', 'Pesan berhasil dihapus.');
    }

    public function bulkDestroy(Request $request)
    {
        $data = $request->validate([
            'ids'   => ['required', 'array', 'min:1'],
            'ids.*' => ['integer', 'exists:contact_messages,id'],
        ], [
            'ids.required' => 'Pilih minimal 1 pesan.',
        ]);

        ContactMessage::whereIn('id', $data['ids'])->delete();

        return redirect()
            ->route('admin.messages.index')
            ->with('success', 'Pesan terpilih berhasil dihapus.');
    }

    // JSON untuk auto-refresh badge lonceng
    public function unreadCount()
    {
        return response()->json([
            'count' => ContactMessage::where('status', 'new')->count(),
        ]);
    }
}
