<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Cv;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class CvController extends Controller
{
    /* =============== INDEX =============== */
    public function index(Request $request)
    {
        $q = (string) $request->query('q', '');

        $cvs = Cv::query()
            ->when($q !== '', fn ($w) =>
                $w->where(fn ($x) =>
                    $x->where('title', 'like', "%{$q}%")
                      ->orWhere('slug', 'like', "%{$q}%")
                      ->orWhere('summary', 'like', "%{$q}%")
                )
            )
            ->latest('id')
            ->paginate(12)
            ->withQueryString();

        return view('backend.cvs.index', compact('cvs', 'q'));
    }

    /* =============== CREATE/STORE =============== */
    public function create()
    {
        $cv = new Cv();
        return view('backend.cvs.create', compact('cv'));
    }

    public function store(Request $request)
    {
        $data = $this->validated($request);

        // Slug unik (pakai input jika ada; fallback dari title)
        $desired      = $request->filled('slug') ? Str::slug($request->slug) : Str::slug($data['title']);
        $data['slug'] = $this->uniqueSlug($desired);

        // Status publish
        $data['published_at'] = $request->boolean('is_published') ? now() : null;

        // Upload file (wajib saat create)
        $file = $request->file('file');
        $data['file_path'] = $file->store('cvs', 'public');

        Cv::create([
            'title'        => $data['title'],
            'slug'         => $data['slug'],
            'summary'      => $data['summary'] ?? null,
            'file_path'    => $data['file_path'],
            'published_at' => $data['published_at'],
        ]);

        return redirect()->route('admin.cvs.index')->with('success', 'CV berhasil dibuat.');
    }

    /* =============== EDIT/UPDATE =============== */
    public function edit(Cv $cv)
    {
        return view('backend.cvs.edit', compact('cv'));
    }

    public function update(Request $request, Cv $cv)
    {
        $data = $this->validated($request, $cv->id);

        if ($request->filled('slug')) {
            $desired = Str::slug($request->slug);
            if ($desired !== $cv->slug) {
                $data['slug'] = $this->uniqueSlug($desired, $cv->id);
            }
        }

        // Ganti file (opsional)
        if ($request->hasFile('file')) {
            if ($cv->file_path) {
                Storage::disk('public')->delete($cv->file_path);
            }
            $data['file_path'] = $request->file('file')->store('cvs', 'public');
        }

        // Status publish
        $data['published_at'] = $request->boolean('is_published') ? now() : null;

        $cv->update([
            'title'        => $data['title'],
            'slug'         => $data['slug'] ?? $cv->slug,
            'summary'      => $data['summary'] ?? $cv->summary,
            'file_path'    => $data['file_path'] ?? $cv->file_path,
            'published_at' => $data['published_at'],
        ]);

        return redirect()->route('admin.cvs.index')->with('success', 'CV berhasil diperbarui.');
    }

    /* =============== DELETE =============== */
    public function destroy(Cv $cv)
    {
        if ($cv->file_path) {
            Storage::disk('public')->delete($cv->file_path);
        }
        $cv->delete();

        return back()->with('success', 'CV dihapus.');
    }

    /* =============== TOGGLE PUBLISH =============== */
    public function publish(Cv $cv)
    {
        $cv->published_at = $cv->published_at ? null : now();
        $cv->save();

        return back()->with('success', $cv->published_at ? 'CV dipublish.' : 'CV diunpublish.');
    }

    /* =============== BULK DELETE =============== */
    public function bulkDestroy(Request $request)
    {
        $ids = array_filter((array) $request->input('ids', []), fn ($v) => is_numeric($v));
        if (empty($ids)) {
            return back()->with('warning', 'Tidak ada item yang dipilih.');
        }

        $cvs = Cv::whereIn('id', $ids)->get();
        foreach ($cvs as $cv) {
            if ($cv->file_path) {
                Storage::disk('public')->delete($cv->file_path);
            }
            $cv->delete();
        }

        return back()->with('success', 'CV terpilih berhasil dihapus.');
    }

    /* =============== VALIDATION & HELPERS =============== */
    private function validated(Request $request, ?int $ignoreId = null): array
    {
        return $request->validate([
            'title'         => ['required', 'string', 'max:200'],
            'slug'          => ['nullable', 'string', 'max:220', Rule::unique('cvs', 'slug')->ignore($ignoreId)],
            'summary'       => ['nullable', 'string', 'max:1000'],
            'is_published'  => ['nullable', 'boolean'],

            // file: wajib saat create; izinkan pdf/doc/docx (pakai 'mimes' agar lebih kompatibel)
            'file'          => [
                $ignoreId ? 'nullable' : 'required',
                'file',
                'mimes:pdf,doc,docx',
                'max:15360', // 15 MB
            ],
        ]);
    }

    private function uniqueSlug(string $base, ?int $ignoreId = null): string
    {
        $base = $base ?: Str::random(8);
        $slug = $base;
        $i = 2;

        while (
            Cv::where('slug', $slug)
                ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
                ->exists()
        ) {
            $slug = $base . '-' . $i++;
        }

        return $slug;
    }
}
