<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Experience;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class ExperienceController extends Controller
{
    public function index(Request $request)
    {
        $q = trim((string) $request->query('q', ''));

        $experiences = Experience::query()
            ->when($q !== '', function ($w) use ($q) {
                $w->where(fn($x) =>
                    $x->where('company', 'like', "%{$q}%")
                      ->orWhere('title', 'like', "%{$q}%")
                      ->orWhere('location', 'like', "%{$q}%")
                );
            })
            ->ordered()
            ->paginate(10)
            ->withQueryString();

        return view('backend.experiences.index', compact('experiences', 'q'));
    }

    public function create()
    {
        return view('backend.experiences.create');
    }

    public function store(Request $request)
    {
        $data = $this->validated($request);
        Experience::create($data);

        return redirect()->route('admin.experiences.index')
            ->with('status', 'Riwayat pekerjaan berhasil dibuat.');
    }

    public function edit(Experience $experience)
    {
        return view('backend.experiences.edit', compact('experience'));
    }

    public function update(Request $request, Experience $experience)
    {
        $data = $this->validated($request, $experience);
        $experience->update($data);

        return redirect()->route('admin.experiences.index')
            ->with('status', 'Riwayat pekerjaan berhasil diperbarui.');
    }

    public function destroy(Experience $experience)
    {
        $experience->delete();
        return back()->with('status', 'Riwayat pekerjaan dihapus.');
    }

    public function toggle(Experience $experience)
    {
        $experience->is_active = ! $experience->is_active;
        $experience->save();

        return back()->with('status', "Riwayat '{$experience->company}' ".($experience->is_active ? 'diaktifkan' : 'dinonaktifkan').'.');
    }

    private function validated(Request $request, ?Experience $experience = null): array
    {
        $data = $request->validate([
            'company'     => ['required', 'string', 'max:150'],
            'title'       => ['required', 'string', 'max:150'],
            'location'    => ['nullable', 'string', 'max:150'],
            'started_at'  => ['required', 'date'],
            'ended_at'    => ['nullable', 'date', 'after_or_equal:started_at'],
            'description' => ['nullable', 'string', 'max:2000'],
            'is_active'   => ['nullable', 'boolean'],
        ]);

        // checkbox: pakai hidden input 0 + checkbox 1 → boolean tepat
        $data['is_active'] = $request->boolean('is_active');

        return $data;
    }
}
