<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Profile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Http\UploadedFile;

class ProfileController extends Controller
{
    public function edit()
    {
        $profile = Profile::first() ?? new Profile();
        return view('backend.profile.edit', compact('profile'));
    }

    public function update(Request $request)
    {
        $profile = Profile::first() ?? new Profile();

        $data = $request->validate([
            'name'         => ['required','string','max:255'],
            'headline'     => ['nullable','string','max:255'],
            'email'        => ['nullable','email:rfc,dns','max:255'],
            'phone'        => ['nullable','string','max:50'],

            'location'     => ['nullable','string','max:255'],
            'lat'          => ['nullable','numeric','between:-90,90'],
            'lng'          => ['nullable','numeric','between:-180,180'],

            'about_short'  => ['nullable','string','max:1000'],
            'about_long'   => ['nullable','string','max:10000'],
            'github'       => ['nullable','url'],
            'linkedin'     => ['nullable','url'],
            'x'            => ['nullable','url'],

            // file
            'avatar'       => ['nullable','image','mimes:jpg,jpeg,png,webp','max:2048'],
            'cv'           => ['nullable','file','mimes:pdf','max:4096'],
        ]);

        if (empty($data['location']) && $request->filled(['lat','lng'])) {
            $data['location'] = sprintf('%s, %s', $request->lat, $request->lng);
        }

        // ===== Avatar: simpan baru, lalu hapus lama (tiban) =====
        if ($request->hasFile('avatar')) {
            $avatarFile = $request->file('avatar');

            // 1) simpan dulu file baru (pakai nama samaran)
            $newAvatarName = $this->obfuscatedName($avatarFile, 'avatar');
            $newAvatarPath = $avatarFile->storeAs('profiles/avatars', $newAvatarName, 'public');

            // 2) hapus file lama jika ada
            if ($profile->avatar_path && Storage::disk('public')->exists($profile->avatar_path)) {
                Storage::disk('public')->delete($profile->avatar_path);
            }

            // 3) update path
            $data['avatar_path'] = $newAvatarPath;
        }

        // ===== CV: simpan baru, lalu hapus lama (tiban) =====
        if ($request->hasFile('cv')) {
            $cvFile = $request->file('cv');

            $newCvName = $this->obfuscatedName($cvFile, 'cv');
            $newCvPath = $cvFile->storeAs('profiles/cv', $newCvName, 'public');

            if ($profile->cv_path && Storage::disk('public')->exists($profile->cv_path)) {
                Storage::disk('public')->delete($profile->cv_path);
            }

            $data['cv_path'] = $newCvPath;
        }

        $profile->fill($data)->save();

        return redirect()
            ->route('admin.profile.edit')
            ->with('status', 'Profil berhasil diperbarui.');
    }

    public function destroyAvatar()
    {
        $profile = Profile::firstOrFail();

        if ($profile->avatar_path && Storage::disk('public')->exists($profile->avatar_path)) {
            Storage::disk('public')->delete($profile->avatar_path);
        }

        $profile->update(['avatar_path' => null]);

        return back()->with('status', 'Avatar dihapus.');
    }

    public function destroyCv()
    {
        $profile = Profile::firstOrFail();

        if ($profile->cv_path && Storage::disk('public')->exists($profile->cv_path)) {
            Storage::disk('public')->delete($profile->cv_path);
        }

        $profile->update(['cv_path' => null]);

        return back()->with('status', 'CV dihapus.');
    }

    /**
     * Nama file berkode samaran:
     * <prefix>_<slug-nama-asli>_<random>.<ext>
     */
    private function obfuscatedName(UploadedFile $file, string $prefix = ''): string
    {
        $origBase = pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME);
        $slugBase = Str::slug(Str::limit($origBase, 60, ''));
        $ext      = strtolower($file->getClientOriginalExtension());
        $rand     = Str::lower(Str::random(10));
        $prefix   = $prefix ? Str::slug($prefix).'_' : '';
        return "{$prefix}{$slugBase}_{$rand}.{$ext}";
    }
}
