<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Profile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ProfilesController extends Controller
{
    public function index(Request $request)
    {
        $q = $request->string('q')->toString();

        $profiles = Profile::query()
            ->when($q, function ($query) use ($q) {
                $query->where(function ($qq) use ($q) {
                    $qq->where('name', 'like', "%{$q}%")
                       ->orWhere('email', 'like', "%{$q}%")
                       ->orWhere('headline', 'like', "%{$q}%");
                });
            })
            ->latest()
            ->paginate(10)
            ->withQueryString();

        $activeId = Profile::where('is_active', true)->value('id');

        return view('backend.profiles.index', compact('profiles', 'q', 'activeId'));
    }

    public function create()
    {
        $profile = new Profile([
            'theme' => 'system',
            'is_public' => true,
            'is_active' => false,
        ]);
        return view('backend.profiles.form', compact('profile'));
    }

    public function store(Request $request)
    {
        $data = $this->validateData($request);

        // simpan file
        if ($request->hasFile('avatar')) {
            $data['avatar_path'] = $request->file('avatar')->store('avatars', 'public');
        }
        if ($request->hasFile('cv')) {
            $data['cv_path'] = $request->file('cv')->store('cv', 'public');
        }

        $profile = Profile::create($data);

        // jika user centang “jadikan aktif”, set aktif tunggal
        if ($request->boolean('make_active')) {
            $this->makeActiveOnly($profile->id);
        }

        return redirect()->route('admin.profiles.index')->with('success', 'Profil dibuat.');
    }

    public function edit(Profile $profile)
    {
        return view('backend.profiles.form', compact('profile'));
    }

    public function update(Request $request, Profile $profile)
    {
        $data = $this->validateData($request, $profile->id);

        // file
        if ($request->hasFile('avatar')) {
            if ($profile->avatar_path) Storage::disk('public')->delete($profile->avatar_path);
            $data['avatar_path'] = $request->file('avatar')->store('avatars', 'public');
        }
        if ($request->hasFile('cv')) {
            if ($profile->cv_path) Storage::disk('public')->delete($profile->cv_path);
            $data['cv_path'] = $request->file('cv')->store('cv', 'public');
        }

        $profile->update($data);

        if ($request->boolean('make_active')) {
            $this->makeActiveOnly($profile->id);
        }

        return redirect()->route('admin.profiles.index')->with('success', 'Profil diperbarui.');
    }

    public function destroy(Profile $profile)
    {
        // hapus file
        if ($profile->avatar_path) Storage::disk('public')->delete($profile->avatar_path);
        if ($profile->cv_path) Storage::disk('public')->delete($profile->cv_path);

        $profile->delete();

        return back()->with('success', 'Profil dihapus.');
    }

    // ---- Aksi khusus ----

    // “Pakai Data”: set satu-satunya yang aktif
    public function use(Profile $profile)
    {
        $this->makeActiveOnly($profile->id);

        return back()->with('success', "Profil '{$profile->name}' sekarang digunakan sebagai profil aktif.");
    }

    // Toggle is_active (tanpa menonaktifkan yang lain)
    public function toggleActive(Profile $profile)
    {
        $profile->is_active = ! $profile->is_active;
        $profile->save();

        return back()->with('success', "Status aktif profil '{$profile->name}' diubah.");
    }

    // Toggle is_public
    public function togglePublic(Profile $profile)
    {
        $profile->is_public = ! $profile->is_public;
        $profile->save();

        return back()->with('success', "Visibilitas profil '{$profile->name}' diubah.");
    }

    // ---- Helper ----

    protected function makeActiveOnly(int $profileId): void
    {
        DB::transaction(function () use ($profileId) {
            Profile::where('is_active', true)->update(['is_active' => false]);
            Profile::where('id', $profileId)->update(['is_active' => true, 'is_public' => true]);
        });
    }

    protected function validateData(Request $request, ?int $id = null): array
    {
        return $request->validate([
            'name'        => ['required','string','max:100'],
            'headline'    => ['nullable','string','max:150'],
            'about_short' => ['nullable','string','max:500'],
            'about_long'  => ['nullable','string'],
            'email'       => [
                'nullable','email','max:255',
                // kalau mau unique email antar profile:
                // Rule::unique('profiles','email')->ignore($id),
            ],
            'phone'       => ['nullable','string','max:50'],
            'location'    => ['nullable','string','max:120'],
            'github'      => ['nullable','url','max:255'],
            'linkedin'    => ['nullable','url','max:255'],
            'x'           => ['nullable','url','max:255'],
            'theme'       => ['required', Rule::in(['light','dark','system'])],
            'is_public'   => ['required','boolean'],
            'is_active'   => ['sometimes','boolean'],

            'avatar'      => ['nullable','image','mimes:jpg,jpeg,png,webp','max:2048'],
            'cv'          => ['nullable','mimes:pdf','max:4096'],

            'make_active' => ['sometimes','boolean'], // checkbox opsional
        ]);
    }
}
