<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Project;
use App\Models\Category;
use App\Models\Skill;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class ProjectController extends Controller
{
    /* ======================= List ======================= */
    public function index(Request $request)
    {
        $q = trim((string) $request->query('q', ''));

        $projects = Project::query()
            ->when($q !== '', fn ($w) =>
                $w->where(fn ($x) =>
                    $x->where('title', 'like', "%{$q}%")
                      ->orWhere('client', 'like', "%{$q}%")
                      ->orWhere('summary', 'like', "%{$q}%")
                )
            )
            ->latest('id')
            ->paginate(12)
            ->withQueryString();

        return view('backend.projects.index', compact('projects', 'q'));
    }

    /* ======================= Create ===================== */
    public function create()
    {
        $project        = new Project();
        $allCategories  = Category::orderBy('name')->get(['id', 'name']);
        $allSkills      = Skill::orderBy('name')->get(['id', 'name']);
        $selectedCats   = [];
        $selectedSkills = [];

        return view('backend.projects.create', compact(
            'project', 'allCategories', 'allSkills', 'selectedCats', 'selectedSkills'
        ));
    }

    public function store(Request $request)
    {
        $data = $this->validated($request);

        // slug unik
        $desired      = $request->filled('slug') ? Str::slug($request->slug) : Str::slug($data['title']);
        $data['slug'] = $this->uniqueSlug($desired);

        // publish opsional
        if ($request->boolean('is_published')) {
            $data['published_at'] = now();
        }

        // ================== Skills & Categories ==================
        // Ambil dari UI (kompatibel dgn 2 nama field: skills[]/skill_ids[] dan categories[]/category_ids[])
        $chosenSkillIds = collect($request->input('skills', $request->input('skill_ids', [])))
            ->map(fn($v) => (int)$v)->filter();
        $chosenCatIds = collect($request->input('categories', $request->input('category_ids', [])))
            ->map(fn($v) => (int)$v)->filter();

        // Skill baru dari skills_new[]
        $newSkillNames = collect($request->input('skills_new', []))
            ->map(fn($v) => trim((string)$v))
            ->filter()
            ->unique();

        $newSkillIds = collect();
        foreach ($newSkillNames as $name) {
            // firstOrCreate agar tidak duplikat
            $skill = Skill::firstOrCreate(
                ['name' => $name],
                [
                    // sesuaikan kolom di tabel skills kamu
                    'slug'      => Str::slug($name),
                    'is_active' => true,
                ]
            );
            $newSkillIds->push($skill->id);
        }

        // Gabungkan & set ke kolom JSON di projects
        $data['skill_ids']    = $chosenSkillIds->merge($newSkillIds)->unique()->values()->all();
        $data['category_ids'] = $chosenCatIds->unique()->values()->all();

        $project = Project::create($data);

        // simpan media
        $this->saveUploadedMedia($project, $request);

        return redirect()->route('admin.projects.index')
            ->with('success', 'Project berhasil dibuat.');
    }

    /* ======================== Edit ====================== */
    public function edit(Project $project)
    {
        $project->loadMissing('media');
        $allCategories  = Category::orderBy('name')->get(['id', 'name']);
        $allSkills      = Skill::orderBy('name')->get(['id', 'name']);
        $selectedCats   = $project->category_ids ?? [];
        $selectedSkills = $project->skill_ids ?? [];

        return view('backend.projects.edit', compact(
            'project', 'allCategories', 'allSkills', 'selectedCats', 'selectedSkills'
        ));
    }

    public function update(Request $request, Project $project)
    {
        $data = $this->validated($request, $project->id);

        if ($request->filled('slug')) {
            $desired = Str::slug($request->slug);
            if ($desired !== $project->slug) {
                $data['slug'] = $this->uniqueSlug($desired, $project->id);
            }
        }

        // toggle publish
        if ($request->boolean('is_published') && !$project->published_at) {
            $data['published_at'] = now();
        } elseif (!$request->boolean('is_published')) {
            $data['published_at'] = null;
        }

        // ================== Skills & Categories ==================
        $chosenSkillIds = collect($request->input('skills', $request->input('skill_ids', [])))
            ->map(fn($v) => (int)$v)->filter();
        $chosenCatIds = collect($request->input('categories', $request->input('category_ids', [])))
            ->map(fn($v) => (int)$v)->filter();

        // Skill baru
        $newSkillNames = collect($request->input('skills_new', []))
            ->map(fn($v) => trim((string)$v))
            ->filter()
            ->unique();

        $newSkillIds = collect();
        foreach ($newSkillNames as $name) {
            $skill = Skill::firstOrCreate(
                ['name' => $name],
                [
                    'slug'      => Str::slug($name),
                    'is_active' => true,
                ]
            );
            $newSkillIds->push($skill->id);
        }

        $data['skill_ids']    = $chosenSkillIds->merge($newSkillIds)->unique()->values()->all();
        $data['category_ids'] = $chosenCatIds->unique()->values()->all();

        $project->update($data);

        // tambah media baru (jika ada)
        $this->saveUploadedMedia($project, $request);

        return redirect()->route('admin.projects.index')
            ->with('success', 'Project berhasil diperbarui.');
    }

    /* ======================= Delete ===================== */
    public function destroy(Project $project)
    {
        foreach ($project->media as $m) {
            if ($m->path)       Storage::disk('public')->delete($m->path);
            if ($m->thumb_path) Storage::disk('public')->delete($m->thumb_path);
        }
        $project->media()->delete();
        $project->delete();

        return back()->with('success', 'Project dihapus.');
    }

    /* =================== Quick Actions ================== */
    public function toggleFeatured(Project $project)
    {
        $project->is_featured = ! $project->is_featured;
        $project->save();

        return back()->with('success', 'Status featured diperbarui.');
    }

    public function publish(Project $project)
    {
        $project->published_at = $project->published_at ? null : now();
        $project->save();

        return back()->with('success', $project->published_at ? 'Project dipublish.' : 'Project diunpublish.');
    }

    /* ===================== Validation ==================== */
    private function validated(Request $request, ?int $ignoreId = null): array
    {
        return $request->validate([
            'title'        => ['required', 'string', 'max:200'],
            'slug'         => ['nullable', 'string', 'max:220', Rule::unique('projects', 'slug')->ignore($ignoreId)],
            'client'       => ['nullable', 'string', 'max:120'],
            'summary'      => ['nullable', 'string', 'max:500'],
            'content'      => ['nullable', 'string'],
            'github_url'   => ['nullable', 'url', 'max:255'],
            'demo_url'     => ['nullable', 'url', 'max:255'],
            'is_featured'  => ['nullable', 'boolean'],
            'published_at' => ['nullable', 'date'],
            'is_published' => ['nullable', 'boolean'],

            // pilihan lama/baru (opsional)
            'skills'          => ['nullable', 'array'],
            'skills.*'        => ['integer'],
            'skill_ids'       => ['nullable', 'array'],
            'skill_ids.*'     => ['integer'],
            'categories'      => ['nullable', 'array'],
            'categories.*'    => ['integer'],
            'category_ids'    => ['nullable', 'array'],
            'category_ids.*'  => ['integer'],
            'skills_new'      => ['nullable', 'array'],
            'skills_new.*'    => ['string', 'max:80'],

            // validasi per file; biarkan 'media' tidak wajib supaya 0..n file
            'media.*'        => ['file', 'mimes:jpg,jpeg,png,webp,mp4,webm', 'max:51200'], // 50MB
            'cover_filename' => ['nullable', 'string', 'max:255'],
        ]);
    }

    private function uniqueSlug(string $base, ?int $ignoreId = null): string
    {
        $base = $base ?: Str::random(8);
        $slug = $base; $i = 2;

        while (Project::where('slug', $slug)
                ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
                ->exists()
        ) {
            $slug = $base.'-'.$i++;
        }
        return $slug;
    }

    /* ================== Media uploader =================== */
    /**
     * Ambil file dari input name="media[]" menggunakan allFiles()
     * sehingga lebih kebal terhadap kasus hasFile() == false.
     */
    private function saveUploadedMedia(Project $project, Request $request): void
    {
        // Diagnostik agar mudah melacak jika file tidak terkirim
        Log::info('saveUploadedMedia:start', [
            'content_type'            => $request->headers->get('content-type'),
            'content_length'          => (int) $request->server('CONTENT_LENGTH'),
            'php.file_uploads'        => ini_get('file_uploads'),
            'php.post_max_size'       => ini_get('post_max_size'),
            'php.upload_max_filesize' => ini_get('upload_max_filesize'),
            'php.max_file_uploads'    => ini_get('max_file_uploads'),
        ]);

        // Ambil SEMUA file yang benar-benar sampai di PHP
        $allFiles   = $request->allFiles();                 // array deep
        $mediaFiles = $allFiles['media'] ?? [];             // cocok utk name="media[]"

        if ($mediaFiles instanceof \Illuminate\Http\UploadedFile) {
            $mediaFiles = [$mediaFiles];
        }

        if (!is_array($mediaFiles) || count($mediaFiles) === 0) {
            // Beri pesan user + catat log
            if (!str_starts_with((string)$request->headers->get('content-type'), 'multipart/form-data')) {
                Log::warning('saveUploadedMedia:no-files (content-type is not multipart/form-data)');
            } elseif ((int) $request->server('CONTENT_LENGTH') === 0) {
                Log::warning('saveUploadedMedia:no-files (CONTENT_LENGTH is 0)');
            } else {
                Log::warning('saveUploadedMedia:no-files (empty media[])');
            }

            session()->flash('warning', 'Tidak ada berkas yang terkirim. Pastikan form memakai enctype="multipart/form-data" dan ukuran file tidak melebihi batas server.');
            return;
        }

        $coverName = (string) $request->input('cover_filename', '');
        $order     = (int) ($project->media()->max('sort_order') ?? 0);
        $newCover  = null;
        $saved     = 0;

        foreach ($mediaFiles as $idx => $file) {
            try {
                if (!$file || !$file->isValid()) {
                    Log::warning('saveUploadedMedia:skip-invalid', [
                        'idx' => $idx,
                        'error' => method_exists($file, 'getError') ? $file->getError() : null,
                    ]);
                    continue;
                }

                $mime = $file->getMimeType() ?: '';
                $kind = str_starts_with($mime, 'video') ? 'video'
                       : (str_starts_with($mime, 'image') ? 'image' : 'file');

                $path = $file->store("project-media/{$project->id}", 'public');

                $media = $project->media()->create([
                    'kind'       => $kind,
                    'path'       => $path,
                    'thumb_path' => null,
                    'caption'    => null,
                    'sort_order' => ++$order,
                    'is_cover'   => false,
                ]);
                $saved++;

                if ($coverName !== '' && $file->getClientOriginalName() === $coverName) {
                    $newCover = $media;
                }

                Log::info('saveUploadedMedia:stored', [
                    'idx'  => $idx,
                    'id'   => $media->id,
                    'path' => $path,
                    'name' => $file->getClientOriginalName(),
                ]);
            } catch (\Throwable $e) {
                Log::error('saveUploadedMedia:exception', [
                    'idx' => $idx,
                    'msg' => $e->getMessage(),
                ]);
                continue;
            }
        }

        // Set cover jika perlu
        if ($saved > 0) {
            if ($newCover) {
                $project->setCover($newCover);
            } elseif (!$project->cover && $project->media()->exists()) {
                $first = $project->media()->orderBy('sort_order')->first();
                if ($first) $project->setCover($first);
            }
        } else {
            session()->flash('warning', 'Tidak ada berkas media yang tersimpan. Periksa ukuran/format file dan batas server.');
        }
    }
}
