<?php
// app/Http/Controllers/Admin/ProjectMediaController.php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Project;
use App\Models\ProjectMedia;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class ProjectMediaController extends Controller
{
    /**
     * Upload 1..n media ke sebuah project.
     * Request:
     *  - project_id   (required)
     *  - media / media[] (file atau array file: image/*, video/*)
     *  - cover_filename (opsional, nama file asli yang dijadikan cover)
     *  - caption[] (opsional, jika kamu kirim caption per file)
     */
    public function store(Request $request)
    {
        // Validasi longgar: izinkan single file ATAU array file
        $request->validate([
            'project_id'     => ['required', 'integer', Rule::exists('projects', 'id')],
            'media'          => ['required'], // bisa single file atau array
            'media.*'        => ['file', 'mimetypes:image/jpeg,image/png,image/webp,video/mp4,video/webm'],
            'cover_filename' => ['nullable', 'string', 'max:255'],
            'caption'        => ['nullable', 'array'],
            'caption.*'      => ['nullable', 'string', 'max:255'],
        ]);

        $project   = Project::findOrFail($request->integer('project_id'));

        // Pastikan selalu array
        $files = $request->file('media');
        if ($files instanceof \Illuminate\Http\UploadedFile) {
            $files = [$files];
        }
        if (!is_array($files) || count($files) === 0) {
            return back()->with('error', 'Tidak ada berkas yang diunggah.');
        }

        $coverName = (string) $request->input('cover_filename', '');
        $orderBase = (int) ($project->media()->max('sort_order') ?? 0);
        $dir       = "project-media/{$project->id}";

        $created   = [];
        $chosenCover = null;

        foreach ($files as $idx => $file) {
            if (!$file || !$file->isValid()) {
                continue;
            }

            $mime = $file->getMimeType() ?: '';
            $kind = str_starts_with($mime, 'video')
                ? 'video'
                : (str_starts_with($mime, 'image') ? 'image' : 'file');

            // Simpan file ke disk public (storage/app/public/...)
            $path = $file->store($dir, 'public');

            $media = $project->media()->create([
                'kind'       => $kind,
                'path'       => $path,
                'thumb_path' => null, // generate thumbnail di tempat lain kalau perlu
                'caption'    => is_array($request->caption ?? null)
                                ? ($request->caption[$idx] ?? null)
                                : null,
                'sort_order' => $orderBase + $idx + 1,
                'is_cover'   => false, // set di bawah
            ]);

            // Tentukan calon cover sesuai cover_filename
            if ($coverName !== '' && $file->getClientOriginalName() === $coverName) {
                $chosenCover = $media;
            }

            $created[] = $media;
        }

        // Atur cover:
        // 1) Jika user memilih cover (dan ada yang match) → set itu.
        // 2) Jika project belum punya media sebelumnya → file batch pertama jadi cover.
        if ($chosenCover) {
            $project->media()->update(['is_cover' => false]);
            $chosenCover->is_cover = true;
            $chosenCover->save();
        } elseif ($orderBase === 0 && !empty($created)) {
            $project->media()->update(['is_cover' => false]);
            $created[0]->is_cover = true;
            $created[0]->save();
        }

        // Balasan fleksibel
        if ($request->wantsJson()) {
            return response()->json([
                'message' => 'Media berhasil diunggah.',
                'items'   => $created,
            ]);
        }

        return back()->with('success', 'Media berhasil diunggah.');
    }

    /**
     * Hapus satu media + file fisiknya.
     */
    public function destroy(ProjectMedia $media)
    {
        $project  = $media->project;
        $wasCover = (bool) $media->is_cover;

        // Hapus file fisik (abaikan jika tidak ada)
        if ($media->path) {
            Storage::disk('public')->delete($media->path);
        }
        if ($media->thumb_path) {
            Storage::disk('public')->delete($media->thumb_path);
        }

        $media->delete();

        // Jika cover terhapus → pilih cover baru (media urutan pertama)
        if ($wasCover) {
            $next = $project->media()->orderBy('sort_order')->first();
            if ($next) {
                $project->media()->update(['is_cover' => false]);
                $next->is_cover = true;
                $next->save();
            }
        }

        return back()->with('success', 'Media dihapus.');
    }

    /**
     * Tandai satu media sebagai cover.
     */
    public function setCover(ProjectMedia $media)
    {
        $project = $media->project;

        $project->media()->update(['is_cover' => false]);
        $media->is_cover = true;
        $media->save();

        return back()->with('success', 'Cover diperbarui.');
    }

    /**
     * Reorder media berdasarkan array id.
     * Request: { order: [mediaId1, mediaId2, ...] }
     */
    public function reorder(Request $request)
    {
        $data = $request->validate([
            'order'   => ['required', 'array', 'min:1'],
            'order.*' => ['integer', Rule::exists('project_media', 'id')],
        ]);

        $items = ProjectMedia::whereIn('id', $data['order'])->get(['id','project_id']);
        if ($items->isEmpty()) {
            return response()->json(['message' => 'Tidak ada media untuk diurutkan'], 422);
        }

        // Pastikan semua media dari project yang sama
        $projectId = $items->first()->project_id;
        if ($items->contains(fn ($m) => $m->project_id !== $projectId)) {
            return response()->json(['message' => 'Order berisi media dari project berbeda'], 422);
        }

        foreach ($data['order'] as $i => $id) {
            ProjectMedia::where('id', $id)->update(['sort_order' => $i + 1]);
        }

        return response()->json(['message' => 'Urutan media diperbarui.']);
    }
}
