<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Skill;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class SkillController extends Controller
{
    public function index(Request $request)
    {
        $q = trim((string) $request->query('q', ''));

        $skills = Skill::query()
            ->when($q !== '', fn ($w) => $w->where('name', 'like', "%{$q}%"))
            ->orderByDesc('id')
            ->paginate(10)
            ->withQueryString();

        return view('backend.skill.index', compact('skills', 'q'));
    }

    public function create()
    {
        return view('backend.skill.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'      => ['required', 'string', 'max:100', Rule::unique('skills', 'name')],
            'icon'      => ['nullable', 'string', 'max:100'],
            // wajib terkirim karena kita kasih hidden input 0
            'is_active' => ['required', 'boolean'],
        ]);

        $data['name'] = trim($data['name']);
        if (array_key_exists('icon', $data)) {
            $data['icon'] = trim((string) $data['icon']);
        }

        // akan jadi true jika checkbox dicentang, false jika tidak
        $data['is_active'] = $request->boolean('is_active');

        Skill::create($data);

        return redirect()->route('admin.skills.index')
            ->with('status', 'Skill berhasil dibuat.');
    }

    public function edit(Skill $skill)
    {
        return view('backend.skill.edit', compact('skill'));
    }

    public function update(Request $request, Skill $skill)
    {
        $data = $request->validate([
            'name'      => ['required', 'string', 'max:100', Rule::unique('skills', 'name')->ignore($skill->id)],
            'icon'      => ['nullable', 'string', 'max:100'],
            // wajib terkirim karena kita kasih hidden input 0
            'is_active' => ['required', 'boolean'],
        ]);

        $data['name'] = trim($data['name']);
        if (array_key_exists('icon', $data)) {
            $data['icon'] = trim((string) $data['icon']);
        }

        $data['is_active'] = $request->boolean('is_active');

        $skill->update($data);

        return redirect()->route('admin.skills.index')
            ->with('status', 'Skill berhasil diperbarui.');
    }

    public function destroy(Skill $skill)
    {
        $skill->delete();
        return back()->with('status', 'Skill dihapus.');
    }

    public function toggle(Skill $skill)
    {
        $skill->is_active = ! $skill->is_active;
        $skill->save();

        return back()->with(
            'status',
            "Skill '{$skill->name}' " . ($skill->is_active ? 'diaktifkan' : 'dinonaktifkan') . '.'
        );
    }

    public function bulkStatus(Request $request)
    {
        $data = $request->validate([
            'ids'    => ['required', 'array', 'min:1'],
            'ids.*'  => ['integer', 'exists:skills,id'],
            'action' => ['required', 'in:activate,deactivate,delete'],
        ]);

        if ($data['action'] === 'delete') {
            Skill::whereIn('id', $data['ids'])->delete();
            return back()->with('status', 'Beberapa skill telah dihapus.');
        }

        $set = $data['action'] === 'activate';
        Skill::whereIn('id', $data['ids'])->update(['is_active' => $set]);

        return back()->with('status', 'Status aktif berhasil diperbarui untuk pilihan.');
    }
}
