<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Video;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class VideoController extends Controller
{
    /* ======================= List ======================= */
    public function index(Request $request)
    {
        $q = (string) $request->query('q', '');

        $videos = Video::query()
            ->when($q !== '', fn ($w) =>
                $w->where(fn ($x) =>
                    $x->where('title', 'like', "%{$q}%")
                      ->orWhere('slug', 'like', "%{$q}%")
                      ->orWhere('description', 'like', "%{$q}%")
                )
            )
            ->latest('id')
            ->paginate(12)
            ->withQueryString();

        return view('backend.videos.index', compact('videos', 'q'));
    }

    /* ======================= Create ===================== */
    public function create()
    {
        $video = new Video();
        return view('backend.videos.create', compact('video'));
    }

    public function store(Request $request)
    {
        $data = $this->validated($request);

        // Slug unik (pakai input jika ada, fallback dari title)
        $desired      = $request->filled('slug') ? Str::slug($request->slug) : Str::slug($data['title']);
        $data['slug'] = $this->uniqueSlug($desired);

        // Status publish dari checkbox
        $data['published_at'] = $request->boolean('is_published') ? now() : null;

        // Upload video (Wajib saat create)
        $file = $request->file('file');
        $data['file_path'] = $file->store('videos', 'public');

        // Auto generate thumbnail (jika ffmpeg tersedia)
        $thumb = $this->generateVideoThumb($data['file_path']); // null jika gagal/ffmpeg tidak ada
        if ($thumb) {
            $data['thumb_path'] = $thumb;
        }

        $video = Video::create([
            'title'        => $data['title'],
            'slug'         => $data['slug'],
            'description'  => $data['description'] ?? null,
            'file_path'    => $data['file_path'],
            'thumb_path'   => $data['thumb_path'] ?? null,
            'published_at' => $data['published_at'],
        ]);

        return redirect()->route('admin.videos.index')
            ->with('success', 'Video berhasil dibuat.');
    }

    /* ======================== Edit ====================== */
    public function edit(Video $video)
    {
        return view('backend.videos.edit', compact('video'));
    }

    public function update(Request $request, Video $video)
    {
        $data = $this->validated($request, $video->id);

        // Slug unik hanya jika diubah/diisi
        if ($request->filled('slug')) {
            $desired = Str::slug($request->slug);
            if ($desired !== $video->slug) {
                $data['slug'] = $this->uniqueSlug($desired, $video->id);
            }
        }

        // Ganti file video (opsional)
        if ($request->hasFile('file')) {
            if ($video->file_path) Storage::disk('public')->delete($video->file_path);
            $data['file_path'] = $request->file('file')->store('videos', 'public');

            // Regenerate thumbnail saat video diganti
            if ($video->thumb_path) Storage::disk('public')->delete($video->thumb_path);
            $thumb = $this->generateVideoThumb($data['file_path']);
            if ($thumb) {
                $data['thumb_path'] = $thumb;
            } else {
                $data['thumb_path'] = null;
            }
        }

        // Status publish dari checkbox
        $data['published_at'] = $request->boolean('is_published') ? now() : null;

        // Update
        $video->update([
            'title'        => $data['title'],
            'slug'         => $data['slug'] ?? $video->slug,
            'description'  => $data['description'] ?? $video->description,
            'file_path'    => $data['file_path'] ?? $video->file_path,
            'thumb_path'   => array_key_exists('thumb_path', $data) ? $data['thumb_path'] : $video->thumb_path,
            'published_at' => $data['published_at'],
        ]);

        return redirect()->route('admin.videos.index')
            ->with('success', 'Video berhasil diperbarui.');
    }

    /* ======================= Delete ===================== */
    public function destroy(Video $video)
    {
        if ($video->file_path) Storage::disk('public')->delete($video->file_path);
        if ($video->thumb_path) Storage::disk('public')->delete($video->thumb_path);
        $video->delete();

        return back()->with('success', 'Video dihapus.');
    }

    /* =================== Quick Toggle =================== */
    public function publish(Video $video)
    {
        $video->published_at = $video->published_at ? null : now();
        $video->save();

        return back()->with('success', $video->published_at ? 'Video dipublish.' : 'Video diunpublish.');
    }

    /* ===================== Validation ==================== */
    private function validated(Request $request, ?int $ignoreId = null): array
    {
        return $request->validate([
            'title'        => ['required', 'string', 'max:200'],
            'slug'         => ['nullable', 'string', 'max:220', Rule::unique('videos', 'slug')->ignore($ignoreId)],
            'description'  => ['nullable', 'string'],

            // Satu kontrol status (checkbox)
            'is_published' => ['nullable', 'boolean'],

            // File video: wajib saat create, opsional saat update
            'file' => [
                $ignoreId ? 'nullable' : 'required',
                'file',
                // tambahkan jenis lain bila perlu
                'mimetypes:video/mp4,video/webm,video/ogg,video/quicktime,video/x-matroska',
                'max:204800', // 200MB
            ],
        ]);
    }

    private function uniqueSlug(string $base, ?int $ignoreId = null): string
    {
        $base = $base ?: Str::random(8);
        $slug = $base;
        $i    = 2;

        while (
            Video::where('slug', $slug)
                ->when($ignoreId, fn ($q) => $q->where('id', '!=', $ignoreId))
                ->exists()
        ) {
            $slug = $base . '-' . $i++;
        }

        return $slug;
    }

    /**
     * Generate thumbnail dari video menggunakan ffmpeg.
     * @param  string $storagePath  path relatif pada disk 'public' (contoh: videos/abcd.mp4)
     * @return string|null          path relatif thumbnail (contoh: videos/thumbs/abcd.jpg) atau null jika gagal
     */
    private function generateVideoThumb(string $storagePath): ?string
    {
        try {
            $inputAbs  = Storage::disk('public')->path($storagePath);
            $fileName  = pathinfo($storagePath, PATHINFO_FILENAME) . '.jpg';
            $relOut    = 'videos/thumbs/' . $fileName;
            $outputAbs = Storage::disk('public')->path($relOut);

            @mkdir(dirname($outputAbs), 0775, true);

            // Ambil frame pada detik ke-2; -q:v 2 = kualitas bagus
            $cmd = sprintf(
                'ffmpeg -ss 00:00:02 -i %s -frames:v 1 -q:v 2 -y %s',
                escapeshellarg($inputAbs),
                escapeshellarg($outputAbs)
            );

            @exec($cmd, $o, $code);

            if ($code === 0 && file_exists($outputAbs)) {
                return $relOut;
            }

            Log::info('FFmpeg thumbnail: skipped or failed', ['code' => $code, 'file' => $storagePath]);
        } catch (\Throwable $e) {
            Log::warning('FFmpeg thumbnail error: '.$e->getMessage());
        }

        return null;
    }
}
