<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;

class LoginController extends Controller
{
    /** Decay per attempt (detik) untuk rate limiter */
    private const ATTEMPT_DECAY = 60;

    public function show()
    {
        return view('backend.auth.login', ['title' => 'Masuk Admin']);
    }

    public function login(Request $request)
    {
        $this->ensureIsNotRateLimited($request);

        // 1) Validasi form + reCAPTCHA v2
        $validated = $request->validate([
            'email'                 => ['required', 'string', 'email:rfc,dns'],
            'password'              => ['required', 'string', 'min:8'],
            'g-recaptcha-response'  => ['required', 'captcha'], // <- penting
        ], [
            'g-recaptcha-response.required' => 'Verifikasi captcha wajib diisi.',
            'g-recaptcha-response.captcha'  => 'Verifikasi captcha gagal. Silakan coba lagi.',
        ]);

        // 2) Autentikasi
        $remember = $request->boolean('remember');

        if (Auth::attempt([
            'email'    => $validated['email'],
            'password' => $validated['password'],
        ], $remember)) {
            $request->session()->regenerate();
            RateLimiter::clear($this->throttleKey($request));

            Log::info('User berhasil login', [
                'email' => $validated['email'],
                'ip'    => $request->ip(),
                'agent' => $request->userAgent(),
            ]);

            return redirect()->intended(route('admin.dashboard'));
        }

        // 3) Gagal login → hit rate limiter
        RateLimiter::hit($this->throttleKey($request), self::ATTEMPT_DECAY);

        Log::warning('Percobaan login gagal', [
            'email' => $validated['email'],
            'ip'    => $request->ip(),
            'agent' => $request->userAgent(),
        ]);

        throw ValidationException::withMessages([
            'email' => __('Email atau password salah.'),
        ]);
    }

    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login');
    }

    protected function ensureIsNotRateLimited(Request $request)
    {
        $key = $this->throttleKey($request);

        if (! RateLimiter::tooManyAttempts($key, 5)) {
            return;
        }

        $seconds = RateLimiter::availableIn($key);

        throw ValidationException::withMessages([
            'email' => trans('Terlalu banyak percobaan login. Coba lagi dalam :seconds detik.', [
                'seconds' => $seconds,
            ]),
        ]);
    }

    protected function throttleKey(Request $request): string
    {
        return Str::lower((string) $request->input('email')) . '|' . $request->ip();
    }
}
