<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rules\Password;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Auth\Events\Registered;
use Illuminate\Validation\ValidationException;

class RegisterController extends Controller
{
    public function show()
    {
        // View: resources/views/backend/auth/register.blade.php
        return view('backend.auth.register', [
            'title' => 'Daftar Akun',
        ]);
    }

    public function store(Request $request)
    {
        $this->ensureIsNotRateLimited($request);

        // Normalisasi input
        $email = Str::lower(trim((string) $request->input('email')));

        $data = $request->validate([
            'name'     => ['required', 'string', 'max:255'],
            'email'    => ['required', 'string', 'email:rfc,dns', 'max:255', 'unique:users,email'],
            'password' => [
                'required',
                'confirmed',
                Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised(),
            ],
            // contoh opsional:
            // 'terms'    => ['accepted'],
        ]);

        // Paksa email hasil normalisasi (menghindari variasi case)
        $data['email'] = $email;

        $user = User::create([
            'name'     => $data['name'],
            'email'    => $data['email'],
            'password' => Hash::make($data['password']),
            // Jika ingin auto-verify, aktifkan baris di bawah (tidak disarankan untuk publik):
            // 'email_verified_at' => now(),
        ]);

        // Trigger email verification (jika model implements MustVerifyEmail)
        event(new Registered($user));

        // Clear limiter bila sukses
        RateLimiter::clear($this->throttleKey($request));

        // ❗ TIDAK auto-login. Arahkan ke halaman login dengan pesan sukses.
        return redirect()
            ->route('login')
            ->with('status', 'Akun berhasil dibuat. Silakan masuk menggunakan email & password Anda.'
                . (method_exists($user, 'hasVerifiedEmail') ? ' Cek email untuk verifikasi akun.' : ''));
    }

    /** Rate limit pendaftaran untuk cegah abuse/brute force */
    protected function ensureIsNotRateLimited(Request $request): void
    {
        // 5 percobaan per 1 menit berdasarkan IP + email
        $maxAttempts = 5;
        $decaySeconds = 60;

        if (! RateLimiter::tooManyAttempts($this->throttleKey($request), $maxAttempts)) {
            return;
        }

        $seconds = RateLimiter::availableIn($this->throttleKey($request));

        throw ValidationException::withMessages([
            'email' => "Terlalu banyak percobaan registrasi. Coba lagi dalam {$seconds} detik.",
        ])->status(429);
    }

    protected function throttleKey(Request $request): string
    {
        return 'register:' . Str::lower((string) $request->input('email')) . '|' . $request->ip();
    }
}
