<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\ContactMessage;
use App\Models\Profile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
// use Illuminate\Support\Facades\Mail; // (opsional) aktifkan kalau mau kirim email
// use App\Mail\ContactMessageReceived; // (opsional) bikin Mailable kalau perlu

class ContactController extends Controller
{
    /**
     * Tampilkan halaman kontak + data profil (email/linkedin/wa) untuk QR di view.
     */
    public function show()
    {
        $profile = Profile::query()->first(); // sesuaikan logika ambil profil aktif kalau perlu
        return view('frontend.contact', compact('profile'));
    }

    /**
     * Terima pesan kontak.
     * - Anti spam dengan RateLimiter (5x per 10 menit per IP+email)
     * - Validasi & normalisasi
     * - Simpan message + meta (ip, user_agent)
     * - Support response JSON untuk AJAX (future chat-friendly)
     */
    public function submit(Request $request)
    {
        // Rate limit
        $key = sprintf('contact:%s:%s', $request->ip(), Str::lower((string) $request->input('email')));
        $maxAttempts   = 5;
        $decaySeconds  = 10 * 60; // 10 menit

        if (RateLimiter::tooManyAttempts($key, $maxAttempts)) {
            $seconds = RateLimiter::availableIn($key);
            $msg = "Terlalu banyak percobaan. Coba lagi dalam {$seconds} detik.";
            return $request->wantsJson()
                ? response()->json(['message' => $msg], 429)
                : back()->withErrors(['rate_limit' => $msg])->withInput();
        }

        // Validasi
        $data = $request->validate([
            'name'    => ['required','string','max:100'],
            'email'   => ['required','email','max:255'],
            'message' => ['required','string','max:2000'],
        ]);

        // Normalisasi sederhana (trim)
        $data['name']    = trim($data['name']);
        $data['email']   = Str::lower(trim($data['email']));
        $data['message'] = trim($data['message']);

        // Simpan meta (pastikan kolomnya ada di tabel & $fillable jika ingin disimpan)
        $meta = [
            'ip_address' => $request->ip(),
            'user_agent' => (string) $request->userAgent(),
            // 'context'    => $request->headers->get('referer'), // opsional
        ];

        // Gabungkan jika model ContactMessage mendukung kolom meta di atas
        $payload = array_merge($data, array_intersect_key($meta, array_flip(['ip_address','user_agent','context'])));

        // Simpan
        $message = ContactMessage::create($payload);

        // Catat satu percobaan sukses (mengurangi sensitivitas block)
        RateLimiter::hit($key, $decaySeconds);

        // (Opsional) kirim email notifikasi ke pemilik profil
        // try {
        //     $to = optional(Profile::query()->first())->email;
        //     if ($to) {
        //         Mail::to($to)->queue(new ContactMessageReceived($message));
        //     }
        // } catch (\Throwable $e) {
        //     // diamkan / log kalau perlu
        // }

        // Response untuk AJAX
        if ($request->wantsJson()) {
            return response()->json([
                'message' => 'Terima kasih, pesan kamu sudah terkirim!',
                'data'    => [
                    'id'      => $message->id,
                    'name'    => $message->name,
                    'email'   => $message->email,
                    'created' => $message->created_at,
                ],
            ], 201);
        }

        // Response HTML standar
        return back()->with('success', 'Terima kasih, pesan kamu sudah terkirim!');
    }
}
