<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\Project;
use Illuminate\Http\Request;

class ProjectController extends Controller
{
    /**
     * Halaman index: tampilkan Terbaru 7 Hari, Paling Banyak Dilihat, dan Semua Project.
     */
    public function index(Request $request)
    {
        $latestWeek = Project::published()
            ->where('published_at', '>=', now()->subDays(7))
            ->with(['media' => fn($q) => $q->where('kind','image')->orderBy('sort_order')])
            ->ordered()
            ->take(8)
            ->get();

        $mostViewed = Project::published()
            ->with(['media' => fn($q) => $q->where('kind','image')->orderBy('sort_order')])
            ->orderByDesc('view_count')->orderByDesc('id')
            ->take(8)
            ->get();

        $projects = Project::published()
            ->with(['media' => fn($q) => $q->where('kind','image')->orderBy('sort_order')])
            ->ordered()
            ->paginate(9);

        return view('frontend.projects.index', compact('latestWeek','mostViewed','projects'));
    }

    /**
     * Detail project: auto-increment view sekali per sesi per project.
     */
    public function show(Request $request, Project $project)
    {
        $project->load(['media' => fn($q) => $q->orderBy('sort_order')]);

        // Naikkan view_count sekali per sesi
        $seen = collect($request->session()->get('seen_projects', []));
        if (!$seen->contains($project->id)) {
            $project->increment('view_count');
            $request->session()->put('seen_projects', $seen->push($project->id)->all());
        }

        return view('frontend.projects.show', compact('project'));
    }

    /**
     * (Opsional) Naikkan view via AJAX, kalau mau dipakai.
     */
    public function incrementView(Project $project)
    {
        $project->increment('view_count');
        return response()->json(['view_count' => $project->view_count]);
    }

    /**
     * Toggle like (session-based).
     */
    public function toggleLike(Request $request, Project $project)
    {
        $liked = collect($request->session()->get('liked_projects', []));
        $isLiked = $liked->contains($project->id);

        if ($isLiked) {
            if ($project->like_count > 0) $project->decrement('like_count');
            $request->session()->put(
                'liked_projects',
                $liked->reject(fn($id) => $id === $project->id)->values()->all()
            );
        } else {
            $project->increment('like_count');
            $request->session()->put('liked_projects', $liked->push($project->id)->all());
        }

        return response()->json([
            'liked' => !$isLiked,
            'like_count' => $project->like_count,
        ]);
    }
}
