<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Str;

class Project extends Model
{
    use HasFactory;

    protected $fillable = [
        'title','slug','client','summary','content',
        'github_url','demo_url','is_featured','published_at',
        'skill_ids','category_ids',
        // counters
        'view_count','like_count',
    ];

    protected $casts = [
        'published_at' => 'datetime',
        'is_featured'  => 'boolean',
        'skill_ids'    => 'array',
        'category_ids' => 'array',
        'view_count'   => 'integer',
        'like_count'   => 'integer',
    ];

    // otomatis ikut saat toArray()/JSON
    protected $appends = ['cover_url','excerpt'];

    /* ================= Relations ================= */
    public function media()
    {
        return $this->hasMany(ProjectMedia::class, 'project_id', 'id')->orderBy('sort_order');
    }

    public function images()
    {
        return $this->hasMany(ProjectMedia::class, 'project_id', 'id')
            ->where('kind','image')->orderBy('sort_order');
    }

    public function cover()
    {
        return $this->hasOne(ProjectMedia::class, 'project_id', 'id')->where('is_cover', true);
    }

    /* ================= Scopes ================= */
    public function scopePublished($q)
    {
        return $q->whereNotNull('published_at')->where('published_at', '<=', now());
    }

    public function scopeOrdered($q)
    {
        return $q->orderByDesc('published_at')->orderByDesc('id');
    }

    public function scopeFeatured($q)
    {
        return $q->where('is_featured', true);
    }

    public function scopeSearch($q, ?string $term)
    {
        $term = trim((string) $term);
        if ($term === '') return $q;

        return $q->where(function ($w) use ($term) {
            $w->where('title', 'like', "%{$term}%")
              ->orWhere('summary', 'like', "%{$term}%")
              ->orWhere('client', 'like', "%{$term}%");
        });
    }

    // terbaru 7 hari terakhir
    public function scopeLatestWeek($q)
    {
        return $q->published()->where('published_at', '>=', now()->subDays(7));
    }

    // urut berdasarkan view terbanyak
    public function scopeMostViewed($q)
    {
        return $q->published()->orderByDesc('view_count')->orderByDesc('id');
    }

    /* ============ Accessors / Helpers ============ */
    public function getRouteKeyName(): string
    {
        return 'slug';
    }

    public function getExcerptAttribute(): string
    {
        $src = $this->summary ?: Str::limit(strip_tags((string) $this->content), 260, '');
        return Str::limit((string) $src, 180);
    }

    public function getCoverUrlAttribute(): string
    {
        $m = $this->cover ?: $this->media()->where('kind', 'image')->first();
        if ($m?->thumb_path) return asset('storage/'.$m->thumb_path);
        if ($m?->path)       return asset('storage/'.$m->path);
        return 'https://picsum.photos/seed/project-cover-'.$this->id.'/800/450';
    }

    protected static function booted(): void
    {
        static::saving(function (Project $p) {
            // slug unik otomatis
            if (!$p->slug && $p->title) {
                $base = Str::slug(Str::limit($p->title, 80, '')) ?: Str::random(8);
                $slug = $base; $i = 2;
                while (static::where('slug', $slug)->whereKeyNot($p->getKey())->exists()) {
                    $slug = $base.'-'.$i++;
                }
                $p->slug = $slug;
            }
        });
    }

    public function setCover(ProjectMedia $media): void
    {
        $this->media()->update(['is_cover' => false]);
        $media->is_cover = true;
        $media->save();
    }

    /**
     * (Opsional) Tambah view sekali per sesi.
     * Panggil dari controller show(): $project->incrementViewOnce();
     */
    public function incrementViewOnce(): void
    {
        try {
            $seen = collect(session('seen_projects', []));
            if (!$seen->contains($this->id)) {
                $this->increment('view_count');
                session(['seen_projects' => $seen->push($this->id)->all()]);
            }
        } catch (\Throwable $e) {
            // fallback aman kalau session tidak tersedia
            $this->increment('view_count');
        }
    }
}
