<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\HtmlString;

class Skill extends Model
{
    use HasFactory;

    protected $table = 'skills';

    protected $fillable = [
        'name',
        'icon',
        'is_active',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    // default nilai jika tidak diisi
    protected $attributes = [
        'is_active' => true,
    ];

    /* ===================== Relationships ===================== */
    public function projects()
    {
        // Default pivot: project_skill (singular, urut alfabetis).
        // Jika pivot kamu bernama lain, gunakan ->withPivot('...') dan ->table('nama_pivot')
        return $this->belongsToMany(Project::class)->withTimestamps();
    }

    /* ===================== Scopes ===================== */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('name');
    }

    /* ===================== Accessors ===================== */
    /**
     * Ikon siap tampil (emoji atau class Font Awesome).
     * - Jika berisi spasi → diasumsikan class (Font Awesome) -> <i class="..."></i>
     * - Jika tidak → anggap emoji/teks pendek.
     * return HtmlString|null (tidak perlu {!! !!} jika pakai {{ }}, karena HtmlString).
     */
    public function getDisplayIconAttribute(): ?HtmlString
    {
        $raw = trim((string) $this->icon);
        if ($raw === '') {
            return null;
        }

        if (str_contains($raw, ' ')) {
            // class FA → aman-kan atribut class
            $safe = e($raw);
            return new HtmlString("<i class=\"{$safe}\"></i>");
        }

        // emoji / teks pendek (escape dulu)
        return new HtmlString(e($raw));
    }

    /* ===================== Mutators (auto-trim) ===================== */
    public function setNameAttribute($value): void
    {
        $this->attributes['name'] = is_string($value) ? trim($value) : $value;
    }

    public function setIconAttribute($value): void
    {
        $this->attributes['icon'] = is_string($value) ? trim($value) : $value;
    }
}
