<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Str;

class Video extends Model
{
    use HasFactory;

    protected $fillable = [
        'title','slug','description','file_path','thumb_path','duration',
        'is_public','published_at','tags'
    ];

    protected $casts = [
        'is_public'     => 'boolean',
        'published_at'  => 'datetime',
        'tags'          => 'array',
    ];

    /** Route binding by slug */
    public function getRouteKeyName(): string
    {
        return 'slug';
    }

    /* ----------------- Accessors (helper URL) ----------------- */

    /** URL absolut ke berkas video di /storage (null kalau kosong) */
    public function getUrlAttribute(): ?string
    {
        return $this->file_path ? asset('storage/'.$this->file_path) : null;
    }

    /** URL absolut ke thumbnail; fallback placeholder bila tidak ada */
    public function getThumbUrlAttribute(): ?string
    {
        if ($this->thumb_path) {
            return asset('storage/'.$this->thumb_path);
        }
        return 'https://picsum.photos/seed/video-'.$this->id.'/640/360';
    }

    /** Convenience flag (optional) */
    public function getIsPublishedAttribute(): bool
    {
        return !is_null($this->published_at) && $this->published_at->isPast();
    }

    /* ----------------------- Scopes ----------------------- */

    public function scopePublished($q)
    {
        return $q->whereNotNull('published_at')->where('published_at', '<=', now());
    }

    public function scopeSearch($q, ?string $term)
    {
        $term = trim((string) $term);
        if ($term === '') return $q;

        return $q->where(function ($w) use ($term) {
            $w->where('title', 'like', "%{$term}%")
              ->orWhere('slug', 'like', "%{$term}%")
              ->orWhere('description', 'like', "%{$term}%");

            // Abaikan jika DB tidak dukung JSON_CONTAINS
            try { $w->orWhereJsonContains('tags', $term); } catch (\Throwable $e) {}
        });
    }

    /* ----------------- Slug unik otomatis ----------------- */

    protected static function booted(): void
    {
        static::saving(function (Video $v) {
            if (!$v->slug && $v->title) {
                $base = Str::slug(Str::limit($v->title, 80, '')) ?: Str::random(8);
                $slug = $base; $i = 2;

                while (static::where('slug', $slug)
                        ->when($v->exists, fn ($q) => $q->whereKeyNot($v->getKey()))
                        ->exists()
                ) {
                    $slug = $base.'-'.$i++;
                }
                $v->slug = $slug;
            }
        });
    }
}
