<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    public function up(): void
    {
        // 1) Tambah kolom JSON di tabel projects
        Schema::table('projects', function (Blueprint $table) {
            if (!Schema::hasColumn('projects','skill_ids')) {
                $table->json('skill_ids')->nullable()->after('demo_url');
            }
            if (!Schema::hasColumn('projects','category_ids')) {
                $table->json('category_ids')->nullable()->after('skill_ids');
            }
        });

        // 2) Backfill dari pivot jika tabel masih ada
        //    - project_skill: project_id, skill_id
        //    - category_project: category_id, project_id (nama umum)
        if (Schema::hasTable('project_skill')) {
            $pairs = DB::table('project_skill')
                ->select('project_id', DB::raw('json_group_array(skill_id) as skill_ids'))
                ->groupBy('project_id')
                ->get();

            foreach ($pairs as $row) {
                // json_group_array di SQLite; Untuk MySQL gunakan GROUP_CONCAT lalu explode
                // Agar aman di semua DB, kita cek dulu apakah bernilai string JSON atau tidak.
                $ids = json_decode($row->skill_ids, true);
                if (!is_array($ids)) {
                    // fallback MySQL/MariaDB: GROUP_CONCAT
                    $ids = DB::table('project_skill')
                        ->where('project_id', $row->project_id)
                        ->pluck('skill_id')
                        ->values()
                        ->all();
                }

                DB::table('projects')
                    ->where('id', $row->project_id)
                    ->update(['skill_ids' => json_encode(array_values(array_unique($ids)))]);
            }
        }

        if (Schema::hasTable('category_project')) {
            $pairs = DB::table('category_project')
                ->select('project_id', DB::raw('GROUP_CONCAT(category_id) as cat_concat'))
                ->groupBy('project_id')
                ->get();

            foreach ($pairs as $row) {
                $ids = collect(explode(',', (string) $row->cat_concat))
                    ->filter()
                    ->map(fn($v)=>(int)$v)
                    ->unique()
                    ->values()
                    ->all();

                DB::table('projects')
                    ->where('id', $row->project_id)
                    ->update(['category_ids' => json_encode($ids)]);
            }
        }

        // 3) Drop pivot tables (opsional – lakukan kalau benar-benar tidak dipakai lagi)
        if (Schema::hasTable('project_skill')) {
            Schema::drop('project_skill');
        }
        if (Schema::hasTable('category_project')) {
            Schema::drop('category_project');
        }
    }

    public function down(): void
    {
        // 1) Recreate pivot tables secara sederhana (jika diperlukan rollback)
        if (!Schema::hasTable('project_skill')) {
            Schema::create('project_skill', function (Blueprint $table) {
                $table->unsignedBigInteger('project_id');
                $table->unsignedBigInteger('skill_id');
                $table->primary(['project_id','skill_id']);
            });
        }
        if (!Schema::hasTable('category_project')) {
            Schema::create('category_project', function (Blueprint $table) {
                $table->unsignedBigInteger('project_id');
                $table->unsignedBigInteger('category_id');
                $table->primary(['project_id','category_id']);
            });
        }

        // 2) Kembalikan data dari kolom JSON ke pivot (sebatas yang ada)
        if (Schema::hasColumn('projects','skill_ids')) {
            $rows = DB::table('projects')->select('id','skill_ids')->whereNotNull('skill_ids')->get();
            foreach ($rows as $r) {
                $ids = json_decode($r->skill_ids, true) ?: [];
                foreach ($ids as $sid) {
                    DB::table('project_skill')->updateOrInsert([
                        'project_id' => $r->id,
                        'skill_id'   => (int)$sid,
                    ], []);
                }
            }
        }
        if (Schema::hasColumn('projects','category_ids')) {
            $rows = DB::table('projects')->select('id','category_ids')->whereNotNull('category_ids')->get();
            foreach ($rows as $r) {
                $ids = json_decode($r->category_ids, true) ?: [];
                foreach ($ids as $cid) {
                    DB::table('category_project')->updateOrInsert([
                        'project_id'  => $r->id,
                        'category_id' => (int)$cid,
                    ], []);
                }
            }
        }

        // 3) Hapus kolom JSON dari projects
        Schema::table('projects', function (Blueprint $table) {
            if (Schema::hasColumn('projects','category_ids')) {
                $table->dropColumn('category_ids');
            }
            if (Schema::hasColumn('projects','skill_ids')) {
                $table->dropColumn('skill_ids');
            }
        });
    }
};
