@extends('frontend.layouts.app')

@section('content')
@php
    /** @var \App\Models\Project $project */
    $images = ($project->media ?? collect())->where('kind','image')->values();

    // Render menggunakan nl2br + auto bold/italic/underline (aman dari XSS)
    $htmlContent = '';
    if (!empty($project->content)) {
        $raw = (string) $project->content;

        // 1) Escape semua HTML agar aman
        $escaped = e($raw);

        // 2) Auto-format: **bold**, *italic*, __underline__
        $patterns = [
            '/\*\*(?=\S)(.+?)(?<=\S)\*\*/s',                   // **bold**
            '/__(?=\S)(.+?)(?<=\S)__/s',                       // __underline__
            '/(?<!\*)\*(?=\S)(.+?)(?<=\S)(?<!\*)\*/s',         // *italic* (hindari bentrok **)
        ];
        $replacements = [
            '<strong>$1</strong>',
            '<u>$1</u>',
            '<em>$1</em>',
        ];
        $formatted = preg_replace($patterns, $replacements, $escaped);

        // 3) Newline -> <br>
        $htmlContent = nl2br($formatted, false);
    }
@endphp

{{-- ===== HERO / HEADER ===== --}}
<header class="relative overflow-hidden">
  <div class="absolute inset-0 bg-gradient-to-br from-indigo-600 via-fuchsia-600 to-rose-500"></div>
  <div class="relative max-w-6xl mx-auto px-4 pt-6 pb-14">
    {{-- Top bar --}}
    <div class="flex items-center justify-between gap-3">
      <nav class="flex items-center gap-2 text-white/90 text-sm">
        <a href="{{ route('home') }}" class="hover:underline">Home</a>
        <span class="opacity-70">/</span>
        <a href="{{ route('projects.index') }}" class="hover:underline">Projects</a>
        <span class="opacity-70">/</span>
        <span class="line-clamp-1">{{ $project->title }}</span>
      </nav>

      <a href="{{ route('projects.index') }}"
         class="inline-flex items-center gap-2 px-3 py-1.5 rounded-xl bg-white/15 text-white border border-white/20 hover:bg-white/25 transition">
        ← Kembali
      </a>
    </div>

    {{-- Title card --}}
    <div class="mt-6 rounded-2xl border border-white/15 bg-white/10 backdrop-blur-md shadow-2xl p-5 text-white">
      <h1 class="text-2xl md:text-3xl font-semibold leading-tight">{{ $project->title }}</h1>

      <div class="mt-3 flex flex-wrap items-center gap-3 text-sm">
        <span class="inline-flex items-center gap-1.5 px-2 py-1 rounded-full bg-white/15 border border-white/20">
          👁 {{ number_format((int)($project->view_count ?? 0)) }}
        </span>
        <span class="inline-flex items-center gap-1.5 px-2 py-1 rounded-full bg-white/15 border border-white/20">
          ❤️ {{ number_format((int)($project->like_count ?? 0)) }}
        </span>
        <span class="inline-flex items-center gap-1.5 px-2 py-1 rounded-full bg-white/15 border border-white/20">
          {{ $project->published_at ? $project->published_at->diffForHumans() : 'Draft' }}
        </span>
        @if($project->client)
          <span class="inline-flex items-center gap-1.5 px-2 py-1 rounded-full bg-white/15 border border-white/20">
            👤 {{ $project->client }}
          </span>
        @endif
      </div>
    </div>
  </div>
</header>

<main class="max-w-6xl mx-auto px-4 -mt-10 pb-14">
  {{-- ===== GALLERY ===== --}}
  <section class="rounded-2xl overflow-hidden border bg-white shadow-lg">
    <div class="relative">
      <div class="slider">
        <div class="slider-rail flex overflow-x-auto snap-x snap-mandatory scroll-smooth no-scrollbar"
             onscroll="window.__dotSync && window.__dotSync(this)"
             data-dots-id="dots-{{ $project->id }}">
          @forelse($images as $img)
            <figure class="snap-start shrink-0 w-full aspect-[16/9] relative bg-zinc-100">
              <img src="{{ asset('storage/'.$img->path) }}" alt="Project image {{ $loop->iteration }}"
                   class="w-full h-full object-cover">
            </figure>
          @empty
            <div class="snap-start shrink-0 w-full aspect-[16/9] grid place-items-center text-zinc-500 bg-zinc-100">
              Tidak ada gambar untuk proyek ini
            </div>
          @endforelse
        </div>

        {{-- Nav --}}
        <button type="button" class="nav left-3"
          onclick="(b=>{const r=b.closest('.slider').querySelector('.slider-rail'); if(r) r.scrollBy({left:-r.clientWidth,behavior:'smooth'});})(this)"></button>
        <button type="button" class="nav right-3"
          onclick="(b=>{const r=b.closest('.slider').querySelector('.slider-rail'); if(r) r.scrollBy({left:r.clientWidth,behavior:'smooth'});})(this)"></button>
      </div>

      {{-- Dots --}}
      <div id="dots-{{ $project->id }}" class="absolute -bottom-2 left-0 right-0 flex justify-center gap-1.5 py-4">
        @for ($i = 0; $i < max(1, $images->count()); $i++)
          <span class="dot{{ $i === 0 ? ' active' : '' }}"></span>
        @endfor
      </div>
    </div>
  </section>

  {{-- ===== META QUICK-LINKS ===== --}}
  <section class="mt-6 grid sm:grid-cols-2 gap-3">
    @if($project->github_url)
      <a href="{{ $project->github_url }}" target="_blank"
         class="group inline-flex items-center justify-center gap-2 rounded-xl border bg-white px-4 py-2 shadow-sm hover:shadow transition">
        <span class="text-indigo-600">GitHub</span>
        <span class="text-zinc-400 group-hover:translate-x-0.5 transition">↗</span>
      </a>
    @endif
    @if($project->demo_url)
      <a href="{{ $project->demo_url }}" target="_blank"
         class="group inline-flex items-center justify-center gap-2 rounded-xl border bg-white px-4 py-2 shadow-sm hover:shadow transition">
        <span class="text-emerald-600">Live Demo</span>
        <span class="text-zinc-400 group-hover:translate-x-0.5 transition">↗</span>
      </a>
    @endif
  </section>

  {{-- ===== CONTENT (scrollable if long) ===== --}}
  <section class="mt-8 rounded-2xl border bg-white shadow-sm p-0">
    @if($htmlContent)
      <div class="content-scroll px-6 py-6 max-h-[65vh] md:max-h-[75vh] overflow-auto" tabindex="0" aria-label="Project content">
        <article class="prose prose-zinc max-w-none">
          {!! $htmlContent !!}
        </article>
      </div>
    @else
      <p class="px-6 py-6 text-zinc-600">Belum ada deskripsi untuk proyek ini.</p>
    @endif
  </section>

  {{-- ===== ACTION BAR ===== --}}
  <section class="mt-8 flex flex-wrap items-center gap-3">
    <a href="{{ route('projects.index') }}"
       class="inline-flex items-center gap-2 px-4 py-2 rounded-xl border bg-white hover:bg-zinc-50 shadow-sm">
      ← Kembali ke daftar
    </a>

    <form method="POST" action="{{ route('projects.like', $project) }}"
          onsubmit="event.preventDefault(); const b=this.querySelector('button'); b.disabled=true;
                   fetch(this.action,{method:'POST',headers:{'X-CSRF-TOKEN':'{{ csrf_token() }}','Accept':'application/json'}})
                     .then(r=>r.json()).then(d=>{b.innerText=(d.liked?'💖 Liked':'🤍 Like')+' ('+d.like_count+')';})
                     .catch(()=>{}).finally(()=>{b.disabled=false;});">
      @csrf
      <button type="submit" class="px-4 py-2 rounded-xl border bg-white hover:bg-rose-50 text-rose-600 shadow-sm">
        🤍 Like ({{ (int)($project->like_count ?? 0) }})
      </button>
    </form>
  </section>
</main>
@endsection

@push('styles')
<style>
  /* --- Slider UI --- */
  .no-scrollbar::-webkit-scrollbar{display:none}
  .no-scrollbar{-ms-overflow-style:none;scrollbar-width:none}
  .slider .nav{
    position:absolute; top:50%; transform:translateY(-50%); z-index:10;
    width:40px; height:40px; display:grid; place-items:center; border-radius:9999px;
    border:1px solid rgba(0,0,0,.06); background:rgba(255,255,255,.96); color:#0f172a;
    box-shadow:0 6px 24px rgba(0,0,0,.08);
  }
  .slider .left-3{ left:.75rem } .slider .right-3{ right:.75rem }
  .dot{ width:8px; height:8px; border-radius:9999px; background:rgba(0,0,0,.25); transition:all .2s }
  .dot.active{ width:22px; background:#0f172a }

  /* --- Prose tune (agar rapi) --- */
  .prose :where(img){ border-radius:.75rem; margin:auto }
  .prose :where(pre){ background:#0f172a; color:#e5e7eb; padding:1rem; border-radius:.75rem; overflow:auto }
  .prose :where(blockquote){ border-left:4px solid #e5e7eb; background:#f8fafc; padding:.25rem .9rem; color:#475569 }
  .prose :where(u){ text-underline-offset: 3px; text-decoration-thickness: from-font; }

  /* --- Scroll area style --- */
  .content-scroll{ scrollbar-gutter: stable; }
  .content-scroll{ overscroll-behavior: contain; }
  .content-scroll::-webkit-scrollbar{ width:10px; }
  .content-scroll::-webkit-scrollbar-thumb{ background:#e5e7eb; border-radius:9999px; }
  .content-scroll:hover::-webkit-scrollbar-thumb{ background:#cbd5e1; }
  .content-scroll::-webkit-scrollbar-track{ background:transparent; }
  .content-scroll{ scrollbar-width: thin; } /* Firefox */
</style>
@endpush

@push('scripts')
<script>
  // Sinkronisasi dot slider
  window.__dotSync = function (rail) {
    const dotsId = rail.getAttribute('data-dots-id');
    if (!dotsId) return;
    const dots = document.getElementById(dotsId)?.querySelectorAll('.dot');
    if (!dots || !dots.length) return;
    const idx = Math.round(rail.scrollLeft / rail.clientWidth);
    dots.forEach((d,i)=> d.classList.toggle('active', i === idx));
  };

  document.addEventListener('DOMContentLoaded', () => {
    document.querySelectorAll('.slider-rail').forEach(rail => window.__dotSync(rail));
  });
</script>
@endpush
