<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Foundation\Auth\EmailVerificationRequest;

/* ========================== Models ========================== */
use App\Models\Skill;
use App\Models\Project;
use App\Models\Experience;
use App\Models\Visitor;
use App\Models\Video;

/* ==================== Public Controllers ==================== */
use App\Http\Controllers\Frontend\ProjectController as FrontProjectController;   // publik
use App\Http\Controllers\Frontend\ContactController;
use App\Http\Controllers\Frontend\AboutController;
use App\Http\Controllers\Frontend\VideoController as FrontVideoController;

/* ===================== Admin Controllers ==================== */
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\ProfileController as AdminProfileController;
use App\Http\Controllers\Admin\ProfilesController;
use App\Http\Controllers\Admin\SkillController;
use App\Http\Controllers\Admin\ExperienceController;
use App\Http\Controllers\Admin\ContactMessageController;
use App\Http\Controllers\Admin\ProjectController as AdminProjectController;     // admin
use App\Http\Controllers\Admin\ProjectMediaController;
use App\Http\Controllers\Admin\VideoController as AdminVideoController;
use App\Http\Controllers\Admin\CvController;

/* =================== Auth (custom) Controllers ============== */
use App\Http\Controllers\Auth\RegisterController;
use App\Http\Controllers\Auth\LoginController;
use App\Http\Controllers\Auth\PasswordConfirmController;

/* ============================================================
|                           Public
|============================================================ */

// Home (kirim section projects + STATS dinamis TANPA Carbon)
Route::get('/', function () {
    $heroVideo = Video::published()->latest('published_at')->first();

    $skills = Skill::active()->ordered()->get();

    // ===== Section Projects =====
    $featured = Project::published()
        ->featured()
        ->with(['media' => fn($q) => $q->where('kind', 'image')->orderBy('sort_order')])
        ->ordered()->take(2)->get();

    $latestWeek = Project::published()
        ->where('published_at', '>=', now()->subDays(7))
        ->with(['media' => fn($q) => $q->where('kind', 'image')->orderBy('sort_order')])
        ->ordered()->take(6)->get();

    $mostViewed = Project::published()
        ->with(['media' => fn($q) => $q->where('kind', 'image')->orderBy('sort_order')])
        ->orderByDesc('view_count')->orderByDesc('id')->take(6)->get();

    // ===== STATS =====
    $projectCount = Project::published()->count();

    // Total pengalaman (bulan) via SQL (pakai kolom started_at / ended_at)
    $totalMonths = (int) DB::table('experiences')
        ->selectRaw('COALESCE(SUM(TIMESTAMPDIFF(MONTH, started_at, IFNULL(ended_at, NOW()))), 0) as m')
        ->value('m');

    $yearsFloat = round($totalMonths / 12, 1);

    // Unique users dari tabel visitors (device unik)
    $uniqueUsers = Visitor::count();

    // Performance grade (heuristik ringan)
    $viewsSum      = (int) Project::sum('view_count');
    $avgViews      = $projectCount ? $viewsSum / max(1, $projectCount) : 0;
    $recentCount7d = $latestWeek->count();

    $score = 60;
    $score += min(20, (int) floor($avgViews / 50)); // rata-rata views → +0..20
    $score += min(10, $recentCount7d * 2);          // aktivitas minggu ini → +0..10
    $score += min(10, $uniqueUsers > 0 ? 10 : 0);   // audience → +0/10
    $score = max(50, min(100, $score));

    $grade = match (true) {
        $score >= 95 => 'A+',
        $score >= 90 => 'A',
        $score >= 85 => 'A-',
        $score >= 80 => 'B+',
        $score >= 75 => 'B',
        $score >= 70 => 'B-',
        $score >= 65 => 'C+',
        $score >= 60 => 'C',
        default      => 'C-',
    };

    return view('frontend.home', compact(
        'skills',
        'featured',
        'latestWeek',
        'mostViewed',
        'projectCount',
        'yearsFloat',
        'uniqueUsers',
        'grade',
        'heroVideo'
    ));
})->name('home');

// Projects (PUBLIC)
Route::get('/projects',                [FrontProjectController::class, 'index'])->name('projects.index');
Route::get('/projects/{project:slug}', [FrontProjectController::class, 'show'])->name('projects.show');
Route::patch('/projects/{project:slug}/view', [FrontProjectController::class, 'incrementView'])->name('projects.view');
Route::post('/projects/{project:slug}/like',  [FrontProjectController::class, 'toggleLike'])->name('projects.like');

// About (PUBLIC)
Route::get('/about', [AboutController::class, 'show'])->name('about');

// Contact (PUBLIC)
Route::get('/contact',  [ContactController::class, 'show'])->name('contact.show');
Route::post('/contact', [ContactController::class, 'submit'])->name('contact.submit');

// ---------- PUBLIC (opsional) ----------
Route::get('/videos', [FrontVideoController::class, 'index'])->name('videos.index');
Route::get('/videos/{video:slug}', [FrontVideoController::class, 'show'])->name('videos.show');

/* ============================================================
|                 Auth (login/register/logout)
|============================================================ */

Route::middleware('guest')->group(function () {
    Route::get('/register',  [RegisterController::class, 'show'])->name('register');
    Route::post('/register', [RegisterController::class, 'store'])
        ->middleware('throttle:3,1')->name('register.submit');

    Route::get('/login',  [LoginController::class, 'show'])->name('login');
    Route::post('/login', [LoginController::class, 'login'])
        ->middleware('throttle:6,1')->name('login.submit');
});

Route::post('/logout', [LoginController::class, 'logout'])
    ->middleware('auth')->name('logout');

/* ============================================================
|                     Email Verification
|============================================================ */

Route::middleware('auth')->group(function () {
    Route::get('/email/verify', fn() => view('auth.verify-email'))->name('verification.notice');

    Route::get('/email/verify/{id}/{hash}', function (EmailVerificationRequest $request) {
        $request->fulfill();
        return redirect()->route('admin.dashboard');
    })->middleware(['signed', 'throttle:6,1'])->name('verification.verify');

    Route::post('/email/verification-notification', function (Request $request) {
        $request->user()->sendEmailVerificationNotification();
        return back()->with('status', 'Tautan verifikasi telah dikirim.');
    })->middleware('throttle:3,1')->name('verification.send');
});

/* ============================================================
|                Password Confirm (middleware)
|============================================================ */

Route::middleware('auth')->group(function () {
    Route::get('/confirm-password',  [PasswordConfirmController::class, 'show'])->name('password.confirm');
    Route::post('/confirm-password', [PasswordConfirmController::class, 'store'])->name('password.confirm.store');
});

/* ============================================================
|                           Admin
|============================================================ */

Route::prefix('admin')
    ->middleware(['auth', 'verified'])
    ->as('admin.')
    ->group(function () {

        // Dashboard
        Route::get('/', [DashboardController::class, 'index'])->name('dashboard');

        // Profile (GET edit pakai password.confirm)
        Route::get('/profile/edit', [AdminProfileController::class, 'edit'])
            ->middleware('password.confirm')->name('profile.edit');
        Route::put('/profile',           [AdminProfileController::class, 'update'])->name('profile.update');
        Route::delete('/profile/avatar', [AdminProfileController::class, 'destroyAvatar'])->name('profile.avatar.destroy');
        Route::delete('/profile/cv',     [AdminProfileController::class, 'destroyCv'])->name('profile.cv.destroy');

        // Profiles CRUD (+ quick actions)
        Route::resource('profiles', ProfilesController::class)->except(['show']);
        Route::post('profiles/{profile}/use',            [ProfilesController::class, 'use'])->name('profiles.use');
        Route::patch('profiles/{profile}/toggle-active', [ProfilesController::class, 'toggleActive'])->name('profiles.toggleActive');
        Route::patch('profiles/{profile}/toggle-public', [ProfilesController::class, 'togglePublic'])->name('profiles.togglePublic');

        // Skills
        Route::post('skills/bulk-status',     [SkillController::class, 'bulkStatus'])->name('skills.bulk-status');
        Route::patch('skills/{skill}/toggle', [SkillController::class, 'toggle'])->name('skills.toggle');
        Route::resource('skills', SkillController::class)->except(['show']);

        // Experiences
        Route::patch('experiences/{experience}/toggle', [ExperienceController::class, 'toggle'])->name('experiences.toggle');
        Route::resource('experiences', ExperienceController::class)->except(['show']);

        // Projects (ADMIN)
        Route::patch('projects/{project}/toggle-featured', [AdminProjectController::class, 'toggleFeatured'])->name('projects.toggle-featured');
        Route::patch('projects/{project}/publish',         [AdminProjectController::class, 'publish'])->name('projects.publish');
        Route::resource('projects', AdminProjectController::class)->except(['show']);

        // Project Media (ADMIN)
        Route::post('media',                 [ProjectMediaController::class, 'store'])->name('media.store');
        Route::delete('media/{media}',       [ProjectMediaController::class, 'destroy'])->name('media.destroy');
        Route::patch('media/{media}/cover',  [ProjectMediaController::class, 'setCover'])->name('media.cover');
        Route::post('media/reorder',         [ProjectMediaController::class, 'reorder'])->name('media.reorder');

        // Inbox ContactMessage
        Route::get('messages',               [ContactMessageController::class, 'index'])->name('messages.index');
        Route::get('messages/{cm}',          [ContactMessageController::class, 'show'])->name('messages.show');
        Route::patch('messages/{cm}',        [ContactMessageController::class, 'update'])->name('messages.update');
        Route::delete('messages/{cm}',       [ContactMessageController::class, 'destroy'])->name('messages.destroy');
        Route::post('messages/bulk-destroy', [ContactMessageController::class, 'bulkDestroy'])->name('messages.bulk-destroy');

        // Endpoint untuk badge lonceng
        Route::get('messages-unread-count',  [ContactMessageController::class, 'unreadCount'])->name('messages.unread-count');

        // Crud VIDEO
        Route::patch('videos/{video}/publish', [AdminVideoController::class, 'publish'])->name('videos.publish');
        Route::resource('videos', AdminVideoController::class)->except(['show']);

        // Crud CV
        Route::resource('cvs', CvController::class)->except(['show']);
        Route::patch('cvs/{cv}/publish', [CvController::class, 'publish'])->name('cvs.publish');
        Route::post('cvs/bulk-destroy', [\App\Http\Controllers\Admin\CvController::class, 'bulkDestroy'])->name('cvs.bulk-destroy');
    });

/* ============================================================
|                          Fallback
|============================================================ */

Route::fallback(fn() => abort(404));
